using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;
using System.Drawing;

namespace Dashboard3
{
	// needed for Today Integration
	public class TodayMsgWnd : Microsoft.WindowsCE.Forms.MessageWindow
	{
		private DateWnd.DateWnd	m_form					= null;				// form for application
		private	IntPtr			m_hwndParent			= (IntPtr) null;	// parent window handle
		private	long			m_lUniqueId				= -1;				// unique id assigned by Dashboard
		private	bool			m_bIsInitialized		= false;			// has Dashboard initialized this app yet?
		private	bool			m_bIsSelected			= false;			// does our application have the selection focus?
		private	int				m_nHeight				= 100;
		private	int				m_nCurrentView			= 0;				// default to first view (0)
		private	int				m_nFontSize				= 8;
		private	Color			m_crText				= Color.Black;
		private Color			m_crHighlight			= Color.Coral;
		private	Color			m_crHighlightText		= Color.Black;

		private	const	int		SHUIM_FONTSIZE_POINT	= 1;				// Application font size (hundredhts of a point) -- buffer is pointer to DWORD
		private	const	int		SHUIM_FONTSIZE_PIXEL	= 2;				// Application font size (in pixels) -- buffer is pointer to DWORD
		private	const	int		SHUIM_FONTSIZE_PERCENTAGE = 3;				// Application font size as percentage of normal -- buffer is pointer to DWORD

		private	const	int		MAX_HEIGHT				= 99999;			// Max Height if want full-screen


		/// <summary>
		/// READ ONLY:
		/// Indicates whether the WM_DASHCUSTOM_INITITEM has been recieved by this
		/// program which means that Dashboard has taken control of it as a module.
		/// </summary>
		public bool IsInitialized
		{	get { return m_bIsInitialized; }
		}


		/// <summary>
		/// READ ONLY:
		/// Indicates whether the module has the selection focus or not.
		/// </summary>
		public bool IsSelected
		{	get { return m_bIsSelected; }
		}

		/// <summary>
		/// Should *always* contain the most up to date Height for the module.
		/// This is used internally to return the height to Dashboard when it
		/// periodically polls the module.
		/// </summary>
		public int Height
		{	get { return m_nHeight; }
			set { m_nHeight = value;}
		}

		/// <summary>
		/// READ ONLY: 
		/// Contains the current view that Dashboard is set to display.
		/// </summary>
		public int CurrentView
		{	get { return m_nCurrentView; }
		}

		/// <summary>
		/// READ ONLY:
		/// Contains the current Font Size of the Today page.
		/// Many devices have a setting that allows the user to adjust
		/// the size of the Today font.  Get the Font size from here
		/// if you want to honor that setting.
		/// </summary>
		public int FontSize
		{	get { return m_nFontSize; }
		}

		/// <summary>
		/// READ ONLY:
		/// Text Color as defined by the Today page.
		/// </summary>
		public Color ColorText
		{	get { return m_crText; }
		}

		/// <summary>
		/// READ ONLY:
		/// Highlight Color as defined by the Today page.
		/// </summary>
		public Color ColorHighlight
		{	get { return m_crHighlight; }
		}

		/// <summary>
		/// READ ONLY:
		/// Highlight Text Color as defined by the Today page.
		/// </summary>
		public Color ColorHighlightText
		{	get { return m_crHighlightText; }
		}


		/// <summary>
		/// Constructor.
		/// Perform class initialization here.
		/// </summary>
		/// <param name="todayForm">C# Form with which to interact</param>
		/// <param name="appName">Name of the application (used to name this window)</param>
		public TodayMsgWnd(DateWnd.DateWnd todayForm, string appName)
		{
			this.Text	= "Today." + appName;

			m_form	= todayForm;
		}


		/// <summary>
		/// Call out to the Today page to paint the background of the window.
		/// Also cache the colors needed for drawing along with the Today screen font size.
		/// </summary>
		/// <param name="hdc"></param>
		public void PaintBackground(IntPtr hdc)
		{
			if (m_hwndParent	!= null)
			{
				// get the text color from the parent
				int		colorText			= (int) SendMessage(m_hwndParent, TodayMessages.TODAYM_GETCOLOR, (IntPtr) TodayMessages.TODAYCOLOR_TEXT			, (IntPtr) m_lUniqueId);
				int		colorHighlight		= (int) SendMessage(m_hwndParent, TodayMessages.TODAYM_GETCOLOR, (IntPtr) TodayMessages.TODAYCOLOR_HIGHLIGHT	, (IntPtr) m_lUniqueId);
				int		colorHighlightText	= (int) SendMessage(m_hwndParent, TodayMessages.TODAYM_GETCOLOR, (IntPtr) TodayMessages.TODAYCOLOR_HIGHLIGHTEDTEXT, (IntPtr) m_lUniqueId);

				// convert the int to the base bytes and extract for the colors
				byte[]	ColorByte			= System.BitConverter.GetBytes(colorText);
				int		ColorRed			= Convert.ToInt32(ColorByte[0]);
				int		ColorGreen			= Convert.ToInt32(ColorByte[1]);
				int		ColorBlue			= Convert.ToInt32(ColorByte[2]);
				m_crText			= Color.FromArgb(ColorRed, ColorGreen, ColorBlue);

				ColorByte			= System.BitConverter.GetBytes(colorHighlight);
				ColorRed			= Convert.ToInt32(ColorByte[0]);
				ColorGreen			= Convert.ToInt32(ColorByte[1]);
				ColorBlue			= Convert.ToInt32(ColorByte[2]);
				m_crHighlight		= Color.FromArgb(ColorRed, ColorGreen, ColorBlue);

				ColorByte			= System.BitConverter.GetBytes(colorHighlightText);
				ColorRed			= Convert.ToInt32(ColorByte[0]);
				ColorGreen			= Convert.ToInt32(ColorByte[1]);
				ColorBlue			= Convert.ToInt32(ColorByte[2]);
				m_crHighlightText	= Color.FromArgb(ColorRed, ColorGreen, ColorBlue);


				if (m_bIsSelected)
				{
					// if the module is selected, don't do anything
					// the main paint routine should handle how it
					// wants to paint the selection
					//SendMessage(m_hwndParent, TodayMessages.WM_DASHCUSTOM_NET_PAINTBKGND, (IntPtr) hdc, (IntPtr) m_lUniqueId);

					// swap the text colors for selected item drawing
					m_crText	= m_crHighlightText;
				}
				else
				{
					//m_form.PopupMessage("m_lUniqueId = " + m_lUniqueId.ToString());
					SendMessage(m_hwndParent, TodayMessages.WM_DASHCUSTOM_NET_PAINTBKGND, (IntPtr) hdc, (IntPtr) m_lUniqueId);
				}

				m_nFontSize			= GetTodayFontSize();
			}
		}


		/// <summary>
		/// Newer devices can control the Today Font Size via a setting in the Settings applet.
		/// This API is how to properly retrieve that setting.
		/// </summary>
		/// <returns>Font size used for the Today screen</returns>
		public int GetTodayFontSize()
		{
			uint	nBuffer		= 0;
			uint	nRequired	= 0;
			uint	hResult		= SHGetUIMetrics(SHUIM_FONTSIZE_PIXEL, out nBuffer, 4, out nRequired);

			return (int) nBuffer;
		}


		/// <summary>
		/// Dashboard has a concept of virtual 'views' where today modules can reside.
		/// This program can reside on one or more 'views' at one time though it will be
		/// handled as the same 'instance' of the program.  Dashboard provides you the
		/// opportunity to allow the user to display different information depending on
		/// what 'view' is being displayed.  For instance, if this were a calendar/tasks
		/// module then the Calendar could be displayed on view 1 while the Tasks could
		/// be displayed on view 2.
		/// 
		/// This is accomplished by calling this function before each paint call 
		/// to determine what you want to paint.
		/// </summary>
		/// <param name="hdc"></param>
		public void GetView()
		{
			if (m_hwndParent	!= null)
			{
				m_nCurrentView	= (int) SendMessage(m_hwndParent, TodayMessages.WM_DASHCUSTOM_GETCURVIEW, (IntPtr) 0, (IntPtr) m_lUniqueId);
			}
		}


		/// <summary>
		/// Request the Selection (focus) on the Today page.  Wait for the WM_TODAYCUSTOM_RECEIVEDSELECTION
		/// message to verify that selection is granted.
		/// </summary>
		public void Select()
		{
			if (m_hwndParent	!= null)
			{
				SendMessage(m_hwndParent, TodayMessages.TODAYM_TOOKSELECTION, (IntPtr) 0, (IntPtr) m_lUniqueId);
			}
		}
		

		/// <summary>
		/// WndProc Override.
		/// Handle custom messages here that impact the Today Page
		/// </summary>
		/// <param name="msg"></param>
		protected override void WndProc(ref Microsoft.WindowsCE.Forms.Message msg)
		{
			bool	bIsHandled	= false;

			switch ((uint)msg.Msg)
			{
				// Initial message sent when Dashboard acknowledges your
				// module on the Today page.  Dashboard's Window handle
				// is sent as the wParam so you know where to send messages.
				// Your (initial) window height should be sent back in the Result.
				// Don't do too much here - Dashboard is waiting for the return
				// result.
				case TodayMessages.WM_DASHCUSTOM_INITITEM:
					{
						m_hwndParent		= (IntPtr) msg.WParam;
						m_lUniqueId			= (long) msg.LParam;
						//m_form.PopupMessage("m_lUniqueId = " + m_lUniqueId.ToString());

						// call the Today initialization for the Form
						m_form.TodayLoad();

						m_bIsInitialized	= true;
						msg.Result			= (IntPtr) m_nHeight;
						bIsHandled			= true;
					}
					break;

				// this message is sent every 20-30 seconds.  
				// Dashboard uses it to determine if your module's
				// height has changed and if it needs to resize your
				// window.  You should never resize your own window.
				case TodayMessages.WM_TODAYCUSTOM_QUERYREFRESHCACHE:
					{
						// return the height on this call
						msg.Result			= (IntPtr) m_nHeight;
						bIsHandled			= true;
					}
					break;

				// this message is sent by Dashboard to get the height of
				// the module just before displaying it on a view
				// (for instance if the view changes).  It can be
				// view specific as the lParam holds the view that the
				// height needs to be returned for so some views can be
				// drawn taller than others.
				// Finally, if the module as displayed within the requested
				// view should be 'auto-sized' - that is to say - take up
				// all available space in the view, then pass back MAX_HEIGHT.
				// Keep in mind that Dashboard does not have scroll bars so
				// the height is still variable depending on Portrait or Landscape.
				case TodayMessages.WM_DASHCUSTOM_GETHEIGHT:
					{
						m_nCurrentView		= (int) msg.LParam;
						msg.Result			= (IntPtr) m_nHeight;
						bIsHandled			= true;
					}
					break;

				// this message is sent whenever Dashboard changes the Theme
				// if your module is sensitive to a theme change then take
				// appropriate action in response to this message.  Most
				// non-Dashboard modules won't care about this message.
				case TodayMessages.WM_DASHCUSTOM_THEMECHANGE:
					{
					}
					break;

				// this message is similar to the one above but indicates
				// that the Dashboard user is changing the language
				// from the General Options in Dashboard.
				case TodayMessages.WM_DASHCUSTOM_SETLANGUAGE:
					{
					}
					break;

				// this message is sent by Dashboard to determine
				// if this program can run in sidebar mode. It is sent
				// just before the module is rendered and if true is returned
				// then this module will be pushed into the sidebar.
				case TodayMessages.WM_DASHCUSTOM_SIDEBAR:
					{
					}
					break;

				// this message is sent whenever the user asks to see the
				// Options dialog for your module.  The View number is passed
				// as the WParam and if you ignore it, then the Options are
				// generic for all views.
				case TodayMessages.WM_DASHCUSTOM_NET_OPTIONDLG:
					{
						// grab the current Dashboard view
						m_nCurrentView		= (int) msg.WParam;

						// say this has been handled
						bIsHandled			= true;
					}
					break;

				// this message is sent whenever the Dashboard
				// wants to show or hide the application.
				case TodayMessages.WM_DASHCUSTOM_NET_SHOW:
					{
						// grab the current Dashboard view
						if ((int) msg.WParam	== 1)
						{
							m_form.Show();
						}
						else
						{
							m_form.Hide();
						}

						// say this has been handled
						bIsHandled			= true;
					}
					break;

				case TodayMessages.WM_DASHCUSTOM_NET_EXIT:
					{
						m_form.TodayClose();

						// say this has been handled
						bIsHandled			= true;
					}
					break;

				#region Selectability Messages
				// this message is sent whenever this module receives the
				// selection focus of the Today page.
				case TodayMessages.WM_TODAYCUSTOM_RECEIVEDSELECTION:
					{
						// return 1 if handled or 0 if the next today item should be called
						m_bIsSelected		= true;
						m_form.Refresh();
						msg.Result			= (IntPtr) 1;
						bIsHandled			= true;
					}
					break;

				// this message is sent whenever this module has lost the
				// selection focus of the Today page.
				case TodayMessages.WM_TODAYCUSTOM_LOSTSELECTION:
					{
						// return code ignored
						m_bIsSelected		= false;
						m_form.Refresh();
						bIsHandled			= true;
					}
					break;

				// this message is sent when the user pushes a navigation button.
				case TodayMessages.WM_TODAYCUSTOM_USERNAVIGATION:
					{
						// return 1 if handled or 0 if the next today item should be called
						m_bIsSelected		= false;
						m_form.Refresh();
						msg.Result			= (IntPtr) 0;
						bIsHandled			= true;
					}
					break;

				// this message is sent when the user pushes an action button.
				case TodayMessages.WM_TODAYCUSTOM_ACTION:
					{
					}
					break;

				#endregion
			}

			if (!bIsHandled)
			{
				// call the base class WndProc for default message handling
				base.WndProc(ref msg);
			}

		}

		[DllImport("coredll.dll")]
		public static extern IntPtr SendMessage(IntPtr hWnd, uint Msg, IntPtr wParam, IntPtr lParam);

		[DllImport("aygshell.dll")]
		private static extern uint SHGetUIMetrics(int uIdType, out uint nBuffer, uint nBufferSize, out uint nRequired);
	}
}

