using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Reflection;
using System.IO;
using System.Drawing.Imaging;
using Microsoft.Win32;

namespace DateWnd
{
	public partial class DateWnd : Form
	{
		/// <summary>
		/// m_mainWndText MUST be set equal to the file name of this application (without the file extension)
		/// or the integration with the Today page will not function properly
		/// </summary>
		public const string					m_mainWndText	= "DatePlugin";

		/// <summary>
		/// m_todayMsgWnd is the main interface between the Today page and this .NETCF application
		/// this is where all the work is performed to make the integration with the Today page work
		/// </summary>
		private Dashboard3.TodayMsgWnd		m_todayMsgWnd	= null;

		/// <summary>
		/// m_iconBitmap is the image icon that is drawn on the today module just in front of the text.
		/// It should be stored as an embedded resource in this program.
		/// </summary>
		private	Bitmap						m_iconBitmap	= null;

		/// <summary>
		/// tracks the last date/time displayed so that we can know when it has changed
		/// </summary>
		private	DateTime					m_displayDate	= DateTime.Now;

		/// <summary>
		/// tracks the area of the screen that holds the date.
		/// used to show the Options dialog when NOT running as a plugin (for testing)
		/// </summary>
		private	Rectangle					m_dateArea		;


		/// <summary>
		/// Form Constructor
		/// Initialize the todayMsgWnd here so that it is valid.
		/// </summary>
		public DateWnd()
		{
			InitializeComponent();

			// initialize the Today page messaging functionality
			m_todayMsgWnd			= new Dashboard3.TodayMsgWnd(this, m_mainWndText);

			// set the initial height that will be used when the form is first set in the Today Page
			// when running inside Dashboard, the C# today module cannot have a height > than the height
			// of the screen (or available space after other modules are drawn) otherwise it will not be
			// displayed.  Today modules typically sacrifice their height and provide some mechanism to
			// scroll through their material.
			// This window is only one line in height so the setting will be minimal initially and then
			// recalculated during the OnPaint call
			m_todayMsgWnd.Height	= 40;
		}


		/// <summary>
		/// Form Load
		/// Setup the integration with Dashboard for the Today page.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OnLoad(object sender, EventArgs e)
		{
			// resize to the full screen width and the default plugin height
			this.Width	= Screen.PrimaryScreen.Bounds.Width;
			this.Height	= m_todayMsgWnd.Height;
			// set the window title so Dashboard can find it
			this.Text	= m_mainWndText;
			this.Update();
		}


		/// <summary>
		/// Perform any Today item initialization here (i.e. start timers).
		/// Do not do any long-running routines, exit as quickly as possible.
		/// </summary>
		public void TodayLoad()
		{
			// TODAY: hide the application it will be reshown when ready by
			// the new parent
			Hide();
			m_timer.Enabled	= true;
		}


		/// <summary>
		/// Handle a call to close the plugin.
		/// Perform any cleanup work here.
		/// </summary>
		public void TodayClose()
		{
			this.Close();
		}


		/// <summary>
		/// Display the Today Options dialog for this module
		/// </summary>
		public void TodayOptions()
		{
			Program.ShowOptions(m_todayMsgWnd.CurrentView);
		}


		/// <summary>
		/// Form Closing
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OnClosed(object sender, EventArgs e)
		{
		}


		/// <summary>
		/// Paint Routine
		/// Most Today modules paint their own window contents and don't rely on controls on the form
		/// because of the transparent nature of the today page.
		/// This sample shows how to do that.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OnPaint(object sender, PaintEventArgs e)
		{
			int				dpiMult		= 1;
			int				widthWindow	= this.Width;
			int				widthImage	= 20;
			int				heightImage	= 20;
			float			pointSize	= 8;

			// TODAY: let the parent draw the background
			if (m_todayMsgWnd.IsInitialized)
			{
				IntPtr		hdc			= e.Graphics.GetHdc();
				Rectangle	clientRect	= new Rectangle(0, 0, this.Width, this.Height);

				// TODAY: Paint the background of this window using the Today background image
				m_todayMsgWnd.PaintBackground(hdc);
				pointSize	= (float)m_todayMsgWnd.FontSize;

				// release the DC that was needed for the background paint call
				e.Graphics.ReleaseHdc(hdc);

				if (m_todayMsgWnd.IsSelected)
				{
					Brush		backBrush	= new SolidBrush(m_todayMsgWnd.ColorHighlight);

					e.Graphics.FillRectangle(backBrush, clientRect);
					backBrush.Dispose();
				}
			}
			else
			{
				pointSize		= m_todayMsgWnd.GetTodayFontSize();
				m_timer.Enabled	= true;
			}

			if (m_iconBitmap	== null)
			{
				// try to load the icon
				Assembly	asm			= Assembly.GetExecutingAssembly();
				string		iconName	= asm.GetName().Name + ".Icon" + e.Graphics.DpiX.ToString() + ".gif";

				m_iconBitmap		= new Bitmap(asm.GetManifestResourceStream(iconName));
			}

			// handle VGA screen sizes properly
			if (e.Graphics.DpiX	== 192)
			{
				dpiMult		 = 2;
				// HACK to allow the Today API to generate the point size accurately on HiDPI mode
				// if anyone has a more "official" means to do this I'd love to know!
				pointSize	 = (float) (pointSize/ (float) 2.5);
				// adjust the size of the image in case there is no image to display
				widthImage	*= dpiMult;
				heightImage	*= dpiMult;
			}

			#region Draw the today image icon
			if (m_iconBitmap	!= null)
			{
				ImageAttributes	iaIcon	= new ImageAttributes();
				Rectangle		rImage	= new Rectangle(0, 0, m_iconBitmap.Width, m_iconBitmap.Height);

				// set bitmap transparency to upper left pixel color
				iaIcon.SetColorKey(m_iconBitmap.GetPixel(0,0), m_iconBitmap.GetPixel(0,0));

				e.Graphics.DrawImage(m_iconBitmap, rImage, 0, 0, m_iconBitmap.Width, m_iconBitmap.Height, GraphicsUnit.Pixel, iaIcon);
				// get the actual image width (this is defaulted if no image is present)
				widthImage	= m_iconBitmap.Width;
				heightImage	= m_iconBitmap.Height;
			}
			#endregion

			#region Draw the date and time
			// ---------------------------------------------------
			// draw the Date and Time
			// ---------------------------------------------------
			Font			fontText	= new Font("Tahoma", pointSize, FontStyle.Bold);
			Brush			brushText	= new SolidBrush(m_todayMsgWnd.ColorText);
			StringFormat	fmtText		= new StringFormat(StringFormatFlags.NoWrap);
			DateTime		dtNow		= DateTime.Now;
			TimeSpan		secs		= new TimeSpan(0, 0, dtNow.Second);
			string			dateText	= dtNow.ToString("dddd, MMMM dd, yyyy");
			string			timeText	= dtNow.ToString("hh:mm tt");
			SizeF			sizeTime	= e.Graphics.MeasureString(timeText, fontText);
			SizeF			sizeDate	= e.Graphics.MeasureString(dateText, fontText);
			// take the larger height (image | text) 
			int				heightText	= ((Convert.ToInt32(sizeTime.Height) + 4) > heightImage) ? (Convert.ToInt32(sizeTime.Height) + 6) : heightImage;
			int				timeLeft	= (int) (widthWindow - (sizeTime.Width + ((int)(sizeTime.Width/2))));
			Rectangle		rectTime	= new Rectangle(timeLeft, 0, (widthWindow - timeLeft - 3), heightText);
			int				widthDate	= (rectTime.Left - widthImage);

			if (sizeDate.Width	> widthDate)
			{
				dateText	= dtNow.ToString("dddd, MMM dd, yyyy");
				sizeDate	= e.Graphics.MeasureString(dateText, fontText);

				if (sizeDate.Width	> widthDate)
				{
					dateText	= dtNow.ToString("ddd, MMM dd, yyyy");
					sizeDate	= e.Graphics.MeasureString(dateText, fontText);
				}
			}

			Rectangle		rectDate	= new Rectangle((widthImage + (2*dpiMult)), 0, widthDate, heightText);

			fmtText.Alignment		= StringAlignment.Near;
			fmtText.LineAlignment	= StringAlignment.Center;
			e.Graphics.DrawString(dateText, fontText, brushText, rectDate, fmtText);

			fmtText.Alignment	= StringAlignment.Far;
			e.Graphics.DrawString(timeText, fontText, brushText, rectTime, fmtText);

			// clean up
			fontText.Dispose();
			brushText.Dispose();
			fmtText.Dispose();

			#endregion

			// update the displayed date for comparison
			m_displayDate			= dtNow - secs;
			m_dateArea				= rectDate;

			// set the today height
			m_todayMsgWnd.Height	= heightText + 4;
		}


		/// <summary>
		/// Timer Event.
		/// Fires once every minute so update the time on the Form
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OnTimerTick(object sender, EventArgs e)
		{
			DateTime	now		= DateTime.Now;
			TimeSpan	secs	= new TimeSpan(0, 0, now.Second);

			// remove seconds from the comparison
			now	-= secs;

			// if date/time has changed
			if (m_displayDate	!= now)
			{
				this.Refresh();
			}
		}


		/// <summary>
		/// FOR DEBUGGING ONLY:
		/// If the form is loaded outside of the Today page container 
		/// (i.e. the MessageWindow is not properly initialized)
		/// then allow the user to close it with a tap
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OnTap(object sender, MouseEventArgs e)
		{
			Point	hitTest	= new Point(e.X, e.Y);

			if (!m_todayMsgWnd.IsInitialized)
			{
				if (HitTest(m_dateArea, hitTest))
				{
					TodayOptions();
				}
				else
				{
					this.Close();
				}
			}
		}


		/// <summary>
		/// FOR DEBUGGING ONLY:
		/// </summary>
		/// <param name="rect"></param>
		/// <param name="ptTest"></param>
		/// <returns></returns>
		private bool HitTest(Rectangle rect, Point ptTest)
		{
			bool	hitInRect	= false;

			if (ptTest.X	>= rect.Left	&&
				ptTest.X	<= rect.Right	  )
			{
				if (ptTest.Y	>= rect.Top		&&
					ptTest.Y	<= rect.Bottom	  )
				{
					hitInRect	= true;
				}
			}

			return hitInRect;
		}


		/// <summary>
		/// FOR DEBUGGING ONLY:
		/// </summary>
		/// <param name="messageText"></param>
		public void PopupMessage(string messageText)
		{
			MessageBox.Show(messageText);
		}
	}
}